const express = require('express');
const nodemailer = require('nodemailer');
const rateLimit = require('express-rate-limit');
const helmet = require('helmet');
const cors = require('cors');
const validator = require('validator');
const xss = require('xss');
const compression = require('compression');

const app = express();
const PORT = 3001;

// ===== CONFIGURATION SECTION - EDIT THESE VALUES =====
const CONFIG = {
  // Server settings
  allowedOrigins: ['http://localhost:8080', 'https://mathenoid.com'],
  
  // Email settings
  email: {
    host: "smtp.gmail.com",
    port: 587,
    secure: false,
    auth: {
      user: "kelvinnyagah045@gmail.com",
      pass: "ttzxjoxovengstka",
    }
  },
  
  // Email addresses - Updated to use your correct email
  fromEmail: 'kelvinnyagah045@gmail.com',
  adminEmail: 'kelvinnyagah045@gmail.com', // Changed to your email
  
  // Rate limiting
  rateLimit: {
    windowMs: 15 * 60 * 1000, // 15 minutes
    maxRequests: 5 // 5 submissions per window
  }
};
// ===== END CONFIGURATION SECTION =====

// Security middleware
app.use(helmet());
app.use(compression());
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// CORS configuration
app.use(cors({
  origin: CONFIG.allowedOrigins,
  credentials: true,
  methods: ['GET', 'POST'],
  allowedHeaders: ['Content-Type', 'Authorization']
}));

// Rate limiting
const contactRateLimit = rateLimit({
  windowMs: CONFIG.rateLimit.windowMs,
  max: CONFIG.rateLimit.maxRequests,
  message: {
    error: 'Too many contact form submissions. Please try again later.',
    retryAfter: '15 minutes'
  },
  standardHeaders: true,
  legacyHeaders: false,
  handler: (req, res) => {
    res.status(429).json({
      success: false,
      error: 'Rate limit exceeded',
      message: 'Too many contact form submissions. Please try again in 15 minutes.'
    });
  }
});

// Email transporter
const transporter = nodemailer.createTransport({
  ...CONFIG.email,
  pool: true,
  maxConnections: 5,
  maxMessages: 100
});

// Verify email configuration on startup
transporter.verify((error, success) => {
  if (error) {
    console.error('❌ Email configuration error:', error.message);
    console.log('💡 Please check your email settings in the CONFIG section');
  } else {
    console.log('✅ Email server is ready to send messages');
  }
});

// Input validation and sanitization
const validateContactForm = (data) => {
  const errors = [];
  const sanitized = {};

  // Required fields validation
  const requiredFields = ['firstName', 'lastName', 'email', 'message'];
  requiredFields.forEach(field => {
    if (!data[field] || typeof data[field] !== 'string' || data[field].trim().length === 0) {
      errors.push(`${field} is required`);
    }
  });

  if (errors.length > 0) return { isValid: false, errors, sanitized };

  // Sanitize and validate individual fields
  sanitized.firstName = xss(validator.escape(data.firstName.trim()));
  sanitized.lastName = xss(validator.escape(data.lastName.trim()));
  sanitized.email = validator.normalizeEmail(data.email.trim().toLowerCase());
  sanitized.phone = data.phone ? xss(validator.escape(data.phone.trim())) : '';
  sanitized.service = data.service ? xss(validator.escape(data.service.trim())) : '';
  sanitized.message = xss(data.message.trim());
  sanitized.budget = data.budget ? xss(validator.escape(data.budget.trim())) : '';

  // Email validation
  if (!validator.isEmail(sanitized.email)) {
    errors.push('Please provide a valid email address');
  }

  // Phone validation (if provided)
  if (sanitized.phone && !validator.isMobilePhone(sanitized.phone, 'any', { strictMode: false })) {
    errors.push('Please provide a valid phone number');
  }

  // Length validations
  if (sanitized.firstName.length > 50) errors.push('First name must be less than 50 characters');
  if (sanitized.lastName.length > 50) errors.push('Last name must be less than 50 characters');
  if (sanitized.message.length > 2000) errors.push('Message must be less than 2000 characters');

  return { isValid: errors.length === 0, errors, sanitized };
};

// Updated Email templates with consistent styling
const createEmailTemplates = (data) => {
  const { firstName, lastName, email, phone, service, message, budget } = data;
  const timestamp = new Date().toLocaleString();
  const messageWordCount = message.split(' ').length;
  const leadId = Date.now().toString().slice(-6);
  
  // Clean, professional admin notification email matching client style
  const adminEmail = {
    from: CONFIG.fromEmail,
    to: CONFIG.adminEmail,
    subject: `🎯 New Lead: ${firstName} ${lastName} - Project Inquiry`,
    html: `
      <div style="font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; max-width: 600px; margin: 0 auto; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border-radius: 15px; overflow: hidden;">
        <div style="padding: 30px; text-align: center;">
          <h1 style="margin: 0; font-size: 32px; font-weight: bold;">🎯 Mathenoid</h1>
          <p style="margin: 10px 0 0 0; font-size: 18px; opacity: 0.9;">New Client Inquiry</p>
        </div>
        
        <div style="padding: 30px; background: white; color: #333;">
          <h2 style="color: #667eea; margin-top: 0;">New Lead Alert! 🚀</h2>
          
          <p>You have received a new project inquiry from a potential client.</p>
          
          <div style="background: #f8f9fa; padding: 20px; border-radius: 10px; margin: 20px 0; border-left: 4px solid #667eea;">
            <h3 style="margin-top: 0; color: #667eea;">👤 Client Information</h3>
            <p style="margin: 5px 0;"><strong>Name:</strong> ${firstName} ${lastName}</p>
            <p style="margin: 5px 0;"><strong>Email:</strong> <a href="mailto:${email}" style="color: #667eea; text-decoration: none;">${email}</a></p>
            ${phone ? `<p style="margin: 5px 0;"><strong>Phone:</strong> <a href="tel:${phone}" style="color: #667eea; text-decoration: none;">${phone}</a></p>` : ''}
            <p style="margin: 5px 0;"><strong>Submitted:</strong> ${timestamp}</p>
            <p style="margin: 5px 0;"><strong>Lead ID:</strong> #${leadId}</p>
          </div>
          
          <div style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; padding: 20px; border-radius: 10px; margin: 20px 0;">
            <h3 style="margin-top: 0;">📋 Project Details</h3>
            ${service ? `<p style="margin: 5px 0;"><strong>Service Requested:</strong> ${service}</p>` : ''}
            ${budget ? `<p style="margin: 5px 0;"><strong>Budget Range:</strong> ${budget}</p>` : ''}
            <p style="margin: 5px 0;"><strong>Message Length:</strong> ${messageWordCount} words</p>
          </div>
          
          <div style="background: #f8f9fa; padding: 20px; border-radius: 10px; margin: 20px 0; border-left: 4px solid #28a745;">
            <h3 style="margin-top: 0; color: #28a745;">💬 Client's Message</h3>
            <div style="background: white; padding: 15px; border-radius: 8px; border: 1px solid #e9ecef; font-style: italic; line-height: 1.6;">
              "${message.replace(/\n/g, '<br>')}"
            </div>
          </div>
          
          <div style="background: #fff3cd; padding: 20px; border-radius: 10px; margin: 20px 0; border-left: 4px solid #ffc107;">
            <h3 style="margin-top: 0; color: #856404;">⚡ Quick Actions</h3>
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 10px; margin-top: 15px;">
  <a href="mailto:${email}?subject=Re: Your Project Inquiry&body=Hi ${firstName},%0A%0AThank you for reaching out! I'm excited about your project..."
     style="display: inline-block; background: linear-gradient(135deg, #667eea, #764ba2); color: white; padding: 12px 20px; text-decoration: none; border-radius: 8px; text-align: center; font-weight: bold; margin-bottom: 4px;">
    📧 Reply to Client
  </a>

  ${phone ? `
  <a href="tel:${phone}"
     style="display: inline-block; background: #28a745; color: white; padding: 12px 20px; text-decoration: none; border-radius: 8px; text-align: center; font-weight: bold; margin-top: 4px;">
    📞 Call Now
  </a>
  ` : ''}
</div>

          </div>
          
          <p style="color: #666; font-size: 14px; text-align: center; margin-top: 30px;">
            <strong>💡 Pro Tip:</strong> Respond within 2 hours to maximize conversion potential!
          </p>
        </div>
        
        <div style="padding: 20px; text-align: center; font-size: 14px; opacity: 0.8;">
          <p style="margin: 0;">Ready to convert this lead? 🎯</p>
          <p style="margin: 5px 0 0 0;">Mathenoid Business System</p>
        </div>
      </div>
    `,
    text: `
NEW LEAD ALERT!

CLIENT INFORMATION:
==================
Name: ${firstName} ${lastName}
Email: ${email}
${phone ? `Phone: ${phone}` : ''}
Lead ID: #${leadId}
Submitted: ${timestamp}

PROJECT DETAILS:
===============
${service ? `Service: ${service}` : ''}
${budget ? `Budget: ${budget}` : ''}
Message Length: ${messageWordCount} words

CLIENT'S MESSAGE:
================
"${message}"

QUICK ACTIONS:
=============
📧 Reply: mailto:${email}
${phone ? `📞 Call: ${phone}` : ''}

💡 Respond within 2 hours to maximize conversion!

---
Mathenoid Business System
    `
  };

  // User auto-reply email (keeping your existing design)
  const userEmail = {
    from: CONFIG.fromEmail,
    to: email,
    subject: '🎉 Thank you for contacting Mathenoid - We\'ll be in touch soon!',
    html: `
      <div style="font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; max-width: 600px; margin: 0 auto; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border-radius: 15px; overflow: hidden;">
        <div style="padding: 30px; text-align: center;">
          <h1 style="margin: 0; font-size: 32px; font-weight: bold;">🚀 Mathenoid</h1>
          <p style="margin: 10px 0 0 0; font-size: 18px; opacity: 0.9;">Transforming Ideas into Digital Reality</p>
        </div>
        
        <div style="padding: 30px; background: white; color: #333;">
          <h2 style="color: #667eea; margin-top: 0;">Hi ${firstName}! 👋</h2>
          
          <p>Thank you for reaching out to us! We're excited about the possibility of working together on your project.</p>
          
          <div style="background: #f8f9fa; padding: 20px; border-radius: 10px; margin: 20px 0; border-left: 4px solid #667eea;">
            <h3 style="margin-top: 0; color: #667eea;">📋 What happens next?</h3>
            <ul style="margin: 0; padding-left: 20px;">
              <li style="margin-bottom: 8px;"><strong>Initial Response:</strong> We'll get back to you within 2 hours during business hours</li>
              <li style="margin-bottom: 8px;"><strong>Discovery Call:</strong> A 30-minute consultation to understand your needs better</li>
              <li><strong>Detailed Proposal:</strong> A comprehensive plan with timeline and pricing</li>
            </ul>
          </div>
          
          <div style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; padding: 20px; border-radius: 10px; margin: 20px 0;">
            <h3 style="margin-top: 0;">🎯 Your Project Summary</h3>
            ${service ? `<p><strong>Service:</strong> ${service}</p>` : ''}
            ${budget ? `<p><strong>Budget Range:</strong> ${budget}</p>` : ''}
            <p><strong>Submitted:</strong> ${timestamp}</p>
          </div>
          
          <p>In the meantime, feel free to check out our latest work and insights on our website.</p>
          
          <div style="text-align: center; margin: 30px 0;">
            <a href="https://mathenoid.com" style="display: inline-block; background: linear-gradient(135deg, #667eea, #764ba2); color: white; padding: 15px 30px; text-decoration: none; border-radius: 50px; font-weight: bold;">
              🌟 Visit Our Website
            </a>
          </div>
          
          <p style="color: #666; font-size: 14px;">
            <strong>Need immediate assistance?</strong><br>
            📧 Email: kelvinnyagah045@gmail.com<br>
            📱 Phone: +254 740 46 7735<br>
            ⏰ Available 24/7 for urgent matters
          </p>
        </div>
        
        <div style="padding: 20px; text-align: center; font-size: 14px; opacity: 0.8;">
          <p style="margin: 0;">Ready to build something amazing together? 🚀</p>
          <p style="margin: 5px 0 0 0;">The Mathenoid Team</p>
        </div>
      </div>
    `,
    text: `
Hi ${firstName}!

Thank you for contacting Mathenoid! We're excited about your project.

What happens next:
1. Initial Response: We'll get back to you within 2 hours
2. Discovery Call: 30-minute consultation
3. Detailed Proposal: Comprehensive plan with timeline and pricing

Your Project Summary:
${service ? `Service: ${service}` : ''}
${budget ? `Budget Range: ${budget}` : ''}
Submitted: ${timestamp}

Contact Information:
Email: kelvinnyagah045@gmail.com
Phone: +254 740 46 7735
Available 24/7 for urgent matters

Best regards,
The Mathenoid Team
    `
  };

  return { adminEmail, userEmail };
};

// Health check endpoint
app.get('/health', (req, res) => {
  res.json({ 
    status: 'healthy', 
    timestamp: new Date().toISOString(),
    emailConfigured: !!CONFIG.email.auth
  });
});

// Contact form submission endpoint
app.post('/api/contact', contactRateLimit, async (req, res) => {
  try {
    console.log('📧 Processing contact form submission...');
    
    // Validate and sanitize input
    const validation = validateContactForm(req.body);
    if (!validation.isValid) {
      return res.status(400).json({
        success: false,
        errors: validation.errors,
        message: 'Please check your input and try again.'
      });
    }

    const { sanitized } = validation;
    
    // Create email templates
    const { adminEmail, userEmail } = createEmailTemplates(sanitized);
    
    // Send emails
    try {
      // Send admin notification
      const adminResult = await transporter.sendMail(adminEmail);
      console.log('✅ Admin email sent:', adminResult.messageId);
      
      // Send user auto-reply
      const userResult = await transporter.sendMail(userEmail);
      console.log('✅ User email sent:', userResult.messageId);
      
      console.log('✅ Contact form processed successfully');
      
      res.json({
        success: true,
        message: 'Thank you for your message! We\'ll get back to you within 2 hours.',
        timestamp: new Date().toISOString()
      });
      
    } catch (emailError) {
      console.error('❌ Email sending failed:', emailError.message);
      
      // Still return success to user but log the error
      res.json({
        success: true,
        message: 'Thank you for your message! We received it and will get back to you soon.',
        warning: 'Email notification may be delayed',
        timestamp: new Date().toISOString()
      });
    }
    
  } catch (error) {
    console.error('❌ Contact form error:', error);
    
    res.status(500).json({
      success: false,
      error: 'Internal server error',
      message: 'Something went wrong. Please try again or contact us directly.'
    });
  }
});

// Error handling middleware
app.use((err, req, res, next) => {
  console.error('❌ Unhandled error:', err);
  res.status(500).json({
    success: false,
    error: 'Internal server error',
    message: 'Something went wrong. Please try again later.'
  });
});

// 404 handler
app.use((req, res) => {
  res.status(404).json({
    success: false,
    error: 'Endpoint not found',
    message: 'The requested endpoint does not exist.'
  });
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('👋 SIGTERM received, shutting down gracefully...');
  transporter.close();
  process.exit(0);
});

process.on('SIGINT', () => {
  console.log('👋 SIGINT received, shutting down gracefully...');
  transporter.close();
  process.exit(0);
});

app.listen(PORT, () => {
  console.log(`🚀 Contact form backend running on port ${PORT}`);
  console.log(`🌍 CORS enabled for: ${CONFIG.allowedOrigins.join(', ')}`);
  console.log('💡 Admin notifications will be sent to: kelvinnyagah045@gmail.com');
});

module.exports = app;